// Sample code that demonstrates using pthread mutex synchronization.
// from the same shared global variable in Static Data.
// Interleaving of reads and writes will sometimes cause printed sum to be less
// than expected.
//
// Remember to compile with -pthread

#define _GNU_SOURCE
#include <unistd.h>
#include <sys/types.h> // necessary for tid
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#define NUM_THREADS 50
#define LOOP_NUM 10000

pthread_mutex_t sum_lock;  // global, accessible by all threads

// NEW: struct to pass arguments to thread_main
typedef struct thd_arg_st {
  int* sum_ptr;
  int num;
} thd_arg;

// increment sum_total LOOP_NUM times in an atomic fashion
void* thread_main(void* arg) {
  // NEW: cast arguments back into struct
  thd_arg* a = (thd_arg*) arg;
  int local_sum = 0;
  for (int i = 0; i < a->num; i++) {
    local_sum++;
  }
  printf("Thread ID: %d with sum total of %d.\n", gettid(), local_sum); 

  pthread_mutex_lock(&sum_lock);
  *a->sum_ptr += local_sum;
  pthread_mutex_unlock(&sum_lock);
  // NEW: delete dynamically-allocated struct
  free(a);
  return NULL;  // return type is a pointer
}


int main(int argc, char** argv) {
  // NEW: made sum_total a local variable
  int sum_total = 0;
  pthread_t thds[NUM_THREADS];  // array of thread ids
  pthread_mutex_init(&sum_lock, NULL);  // initialize mutex to default

  // create threads to run thread_main()
  for (int i = 0; i < NUM_THREADS; i++) {
    thd_arg* args = malloc(sizeof(thd_arg));
    args->sum_ptr = &sum_total;
    args->num = LOOP_NUM;
    if (pthread_create(&thds[i], NULL, &thread_main, args) != 0) {
      fprintf(stderr, "pthread_create failed\n");
    }
  }

  // wait for all child threads to finish
  // (children may terminate out of order, but cleans up in order)
  for (int i = 0; i < NUM_THREADS; i++) {
    if (pthread_join(thds[i], NULL) != 0) {
      fprintf(stderr, "pthread_join failed\n");
    }
  }

  // print out the final sum (expecting NUM_THREADS * LOOP_NUM)
  printf("The ultimate sum total is %d\n", sum_total);

  pthread_mutex_destroy(&sum_lock);  // destroy the mutex to clean up
  return EXIT_SUCCESS;
}
