// Sample code that demonstrates using pthread mutex synchronization.
// from the same shared global variable in Static Data.
// Interleaving of reads and writes will sometimes cause printed sum to be less
// than expected.
//
// Remember to compile with -pthread

#define _GNU_SOURCE
#include <unistd.h>
#include <sys/types.h> // necessary for tid
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>


#define NUM_THREADS 50
#define LOOP_NUM 10000

int sum_total = 0;

pthread_mutex_t sum_lock;  // global, accessible by all threads

// increment sum_total LOOP_NUM times in an atomic fashion
void* thread_main(void* arg) {
  pthread_mutex_lock(&sum_lock);
  //make sure to compile with the -01 flag
  for (int i = 0; i < LOOP_NUM; i++) {
    sum_total++;
  }
  pthread_mutex_unlock(&sum_lock);

  printf("Thread ID: %d with sum total of %d.\n", gettid(), sum_total); 
  return NULL;  // return type is a pointer
}


int main(int argc, char** argv) {
  pthread_t thds[NUM_THREADS];  // array of thread ids
  pthread_mutex_init(&sum_lock, NULL);  // initialize mutex to default

  // create threads to run thread_main()
  for (int i = 0; i < NUM_THREADS; i++) {
    if (pthread_create(&thds[i], NULL, &thread_main, NULL) != 0) {
      fprintf(stderr, "pthread_create failed\n");
    }
  }

  // wait for all child threads to finish
  // (children may terminate out of order, but cleans up in order)
  for (int i = 0; i < NUM_THREADS; i++) {
    if (pthread_join(thds[i], NULL) != 0) {
      fprintf(stderr, "pthread_join failed\n");
    }
  }

  // print out the final sum (expecting NUM_THREADS * LOOP_NUM)
  printf("The ultimate sum total is %d\n", sum_total);

  pthread_mutex_destroy(&sum_lock);  // destroy the mutex to clean up
  return EXIT_SUCCESS;
}
